/*
 * Shared data between exp, exp2 and pow.
 *
 * Copyright (c) 2018-2023, Arm Limited.
 * SPDX-License-Identifier: MIT OR Apache-2.0 WITH LLVM-exception
 */

#include "math_config.h"

#define N (1 << EXP_TABLE_BITS)

const struct exp_data __exp_data = {
// N/ln2
.invln2N = 0x1.71547652b82fep0 * N,
.invlog10_2N = 0x1.a934f0979a371p1 * N,
// -ln2/N
#if N == 64
.negln2hiN = -0x1.62e42fefa0000p-7,
.negln2loN = -0x1.cf79abc9e3b3ap-46,
#elif N == 128
.negln2hiN = -0x1.62e42fefa0000p-8,
.negln2loN = -0x1.cf79abc9e3b3ap-47,
#elif N == 256
.negln2hiN = -0x1.62e42fefc0000p-9,
.negln2loN = 0x1.c610ca86c3899p-45,
#elif N == 512
.negln2hiN = -0x1.62e42fef80000p-10,
.negln2loN = -0x1.1cf79abc9e3b4p-45,
#endif
.neglog10_2hiN = -0x1.3441350ap-2 / N,
.neglog10_2loN = 0x1.0c0219dc1da99p-39 / N,
// Used for rounding when !TOINT_INTRINSICS
#if EXP_USE_TOINT_NARROW
.shift = 0x1800000000.8p0,
#else
.shift = 0x1.8p52,
#endif
// exp polynomial coefficients.
.poly = {
#if N == 64 && EXP_POLY_ORDER == 5 && !EXP_POLY_WIDE
// abs error: 1.5543*2^-60
// ulp error: 0.529 (0.533 without fma)
// if |x| < ln2/128+eps
// abs error if |x| < ln2/64: 1.7157*2^-50
0x1.fffffffffdbcdp-2,
0x1.555555555444cp-3,
0x1.555573c6a9f7dp-5,
0x1.1111266d28935p-7,
#elif N == 64 && EXP_POLY_ORDER == 6 && EXP_POLY_WIDE
// abs error: 1.6735*2^-64
// ulp error: 0.518 (0.522 without fma)
// if |x| < ln2/64
0x1.5555555548f9ap-3,
0x1.555555554bf5dp-5,
0x1.11115b75f0f4dp-7,
0x1.6c171a6b6303ep-10,
#elif N == 128 && EXP_POLY_ORDER == 5 && !EXP_POLY_WIDE
// abs error: 1.555*2^-66
// ulp error: 0.509 (0.511 without fma)
// if |x| < ln2/256+eps
// abs error if |x| < ln2/256+0x1p-15: 1.09*2^-65
// abs error if |x| < ln2/128: 1.7145*2^-56
0x1.ffffffffffdbdp-2,
0x1.555555555543cp-3,
0x1.55555cf172b91p-5,
0x1.1111167a4d017p-7,
#elif N == 128 && EXP_POLY_ORDER == 5 && EXP_POLY_WIDE
// abs error: 1.5542*2^-60
// ulp error: 0.521 (0.523 without fma)
// if |x| < ln2/128
0x1.fffffffffdbcep-2,
0x1.55555555543c2p-3,
0x1.555573c64f2e3p-5,
0x1.111126b4eff73p-7,
#elif N == 128 && EXP_POLY_ORDER == 6 && EXP_POLY_WIDE
// abs error: 1.6861*2^-71
// ulp error: 0.509 (0.511 without fma)
// if |x| < ln2/128
0x1.55555555548fdp-3,
0x1.555555555658fp-5,
0x1.111123a859bb6p-7,
0x1.6c16ba6920cabp-10,
#elif N == 256 && EXP_POLY_ORDER == 4 && !EXP_POLY_WIDE
// abs error: 1.43*2^-58
// ulp error: 0.549 (0.550 without fma)
// if |x| < ln2/512
0x1p0, // unused
0x1.fffffffffffd4p-2,
0x1.5555571d6ef9p-3,
0x1.5555576a5adcep-5,
#elif N == 256 && EXP_POLY_ORDER == 5 && EXP_POLY_WIDE
// abs error: 1.5547*2^-66
// ulp error: 0.505 (0.506 without fma)
// if |x| < ln2/256
0x1.ffffffffffdbdp-2,
0x1.555555555543cp-3,
0x1.55555cf16e1edp-5,
0x1.1111167a4b553p-7,
#elif N == 512 && EXP_POLY_ORDER == 4 && !EXP_POLY_WIDE
// abs error: 1.4300*2^-63
// ulp error: 0.504
// if |x| < ln2/1024
// abs error if |x| < ln2/512: 1.0689*2^-55
0x1p0, // unused
0x1.ffffffffffffdp-2,
0x1.555555c75bb6p-3,
0x1.555555dec04a8p-5,
#endif
},
.exp2_shift = 0x1.8p52 / N,
// exp2 polynomial coefficients.
.exp2_poly = {
#if N == 64 && EXP2_POLY_ORDER == 6 && EXP2_POLY_WIDE
// abs error: 1.3054*2^-63
// ulp error: 0.515
// if |x| < 1/64
0x1.62e42fefa39efp-1,
0x1.ebfbdff82c58fp-3,
0x1.c6b08d7045cf1p-5,
0x1.3b2ab6fb8fd0ep-7,
0x1.5d884afec48d7p-10,
0x1.43097dc684ae1p-13,
#elif N == 128 && EXP2_POLY_ORDER == 5 && !EXP2_POLY_WIDE
// abs error: 1.2195*2^-65
// ulp error: 0.507 (0.511 without fma)
// if |x| < 1/256
// abs error if |x| < 1/128: 1.9941*2^-56
0x1.62e42fefa39efp-1,
0x1.ebfbdff82c424p-3,
0x1.c6b08d70cf4b5p-5,
0x1.3b2abd24650ccp-7,
0x1.5d7e09b4e3a84p-10,
#elif N == 256 && EXP2_POLY_ORDER == 5 && EXP2_POLY_WIDE
// abs error: 1.2195*2^-65
// ulp error: 0.504 (0.508 without fma)
// if |x| < 1/256
0x1.62e42fefa39efp-1,
0x1.ebfbdff82c424p-3,
0x1.c6b08d70cf4b5p-5,
0x1.3b2abd24650ccp-7,
0x1.5d7e09b4e3a84p-10,
#elif N == 512 && EXP2_POLY_ORDER == 4 && !EXP2_POLY_WIDE
// abs error: 1.4411*2^-64
// ulp error: 0.5024 (0.5063 without fma)
// if |x| < 1/1024
// abs error if |x| < 1/512: 1.9430*2^-56
0x1.62e42fefa39ecp-1,
0x1.ebfbdff82c58bp-3,
0x1.c6b08e46de41fp-5,
0x1.3b2ab786ee1dap-7,
#endif
},
.exp10_poly = {
#if EXP10_POLY_WIDE
/* Range is wider if using shift-based reduction: coeffs generated
   using Remez in [-log10(2)/128, log10(2)/128 ].  */
0x1.26bb1bbb55515p1,
0x1.53524c73cd32bp1,
0x1.0470591e1a108p1,
0x1.2bd77b12fe9a8p0,
0x1.14289fef24b78p-1
#else
/* Coeffs generated using Remez in [-log10(2)/256, log10(2)/256 ].  */
0x1.26bb1bbb55516p1,
0x1.53524c73ce9fep1,
0x1.0470591ce4b26p1,
0x1.2bd76577fe684p0,
0x1.1446eeccd0efbp-1
#endif
},
// 2^(k/N) ~= H[k]*(1 + T[k]) for int k in [0,N)
// tab[2*k] = asuint64(T[k])
// tab[2*k+1] = asuint64(H[k]) - (k << 52)/N
.tab = {
#if N == 64
0x0, 0x3ff0000000000000,
0xbc7160139cd8dc5d, 0x3fefec9a3e778061,
0x3c8cd2523567f613, 0x3fefd9b0d3158574,
0x3c60f74e61e6c861, 0x3fefc74518759bc8,
0x3c979aa65d837b6d, 0x3fefb5586cf9890f,
0x3c3ebe3d702f9cd1, 0x3fefa3ec32d3d1a2,
0xbc9556522a2fbd0e, 0x3fef9301d0125b51,
0xbc91c923b9d5f416, 0x3fef829aaea92de0,
0xbc801b15eaa59348, 0x3fef72b83c7d517b,
0x3c8b898c3f1353bf, 0x3fef635beb6fcb75,
0x3c9aecf73e3a2f60, 0x3fef54873168b9aa,
0x3c8a6f4144a6c38d, 0x3fef463b88628cd6,
0x3c968efde3a8a894, 0x3fef387a6e756238,
0x3c80472b981fe7f2, 0x3fef2b4565e27cdd,
0x3c82f7e16d09ab31, 0x3fef1e9df51fdee1,
0x3c8b3782720c0ab4, 0x3fef1285a6e4030b,
0x3c834d754db0abb6, 0x3fef06fe0a31b715,
0x3c8fdd395dd3f84a, 0x3feefc08b26416ff,
0xbc924aedcc4b5068, 0x3feef1a7373aa9cb,
0xbc71d1e83e9436d2, 0x3feee7db34e59ff7,
0x3c859f48a72a4c6d, 0x3feedea64c123422,
0xbc58a78f4817895b, 0x3feed60a21f72e2a,
0x3c4363ed60c2ac11, 0x3feece086061892d,
0x3c6ecce1daa10379, 0x3feec6a2b5c13cd0,
0x3c7690cebb7aafb0, 0x3feebfdad5362a27,
0xbc8f94340071a38e, 0x3feeb9b2769d2ca7,
0xbc78dec6bd0f385f, 0x3feeb42b569d4f82,
0x3c93350518fdd78e, 0x3feeaf4736b527da,
0x3c9063e1e21c5409, 0x3feeab07dd485429,
0x3c9432e62b64c035, 0x3feea76f15ad2148,
0xbc8c33c53bef4da8, 0x3feea47eb03a5585,
0xbc93cedd78565858, 0x3feea23882552225,
0xbc93b3efbf5e2228, 0x3feea09e667f3bcd,
0xbc6367efb86da9ee, 0x3fee9fb23c651a2f,
0xbc781f647e5a3ecf, 0x3fee9f75e8ec5f74,
0xbc8619321e55e68a, 0x3fee9feb564267c9,
0xbc7b32dcb94da51d, 0x3feea11473eb0187,
0x3c65ebe1abd66c55, 0x3feea2f336cf4e62,
0xbc9369b6f13b3734, 0x3feea589994cce13,
0xbc94d450d872576e, 0x3feea8d99b4492ed,
0x3c8db72fc1f0eab4, 0x3feeace5422aa0db,
0x3c7bf68359f35f44, 0x3feeb1ae99157736,
0xbc5da9b88b6c1e29, 0x3feeb737b0cdc5e5,
0xbc92434322f4f9aa, 0x3feebd829fde4e50,
0x3c71affc2b91ce27, 0x3feec49182a3f090,
0xbc87c50422622263, 0x3feecc667b5de565,
0xbc91bbd1d3bcbb15, 0x3feed503b23e255d,
0x3c8469846e735ab3, 0x3feede6b5579fdbf,
0x3c8c1a7792cb3387, 0x3feee89f995ad3ad,
0xbc55c3d956dcaeba, 0x3feef3a2b84f15fb,
0xbc68d6f438ad9334, 0x3feeff76f2fb5e47,
0x3c74ffd70a5fddcd, 0x3fef0c1e904bc1d2,
0x3c736eae30af0cb3, 0x3fef199bdd85529c,
0x3c84e08fd10959ac, 0x3fef27f12e57d14b,
0x3c676b2c6c921968, 0x3fef3720dcef9069,
0xbc8fad5d3ffffa6f, 0x3fef472d4a07897c,
0x3c74a385a63d07a7, 0x3fef5818dcfba487,
0x3c8e5a50d5c192ac, 0x3fef69e603db3285,
0xbc82d52107b43e1f, 0x3fef7c97337b9b5f,
0x3c74b604603a88d3, 0x3fef902ee78b3ff6,
0xbc8ff7128fd391f0, 0x3fefa4afa2a490da,
0x3c8ec3bc41aa2008, 0x3fefba1bee615a27,
0x3c8a64a931d185ee, 0x3fefd0765b6e4540,
0x3c77893b4d91cd9d, 0x3fefe7c1819e90d8,
#elif N == 128
0x0, 0x3ff0000000000000,
0x3c9b3b4f1a88bf6e, 0x3feff63da9fb3335,
0xbc7160139cd8dc5d, 0x3fefec9a3e778061,
0xbc905e7a108766d1, 0x3fefe315e86e7f85,
0x3c8cd2523567f613, 0x3fefd9b0d3158574,
0xbc8bce8023f98efa, 0x3fefd06b29ddf6de,
0x3c60f74e61e6c861, 0x3fefc74518759bc8,
0x3c90a3e45b33d399, 0x3fefbe3ecac6f383,
0x3c979aa65d837b6d, 0x3fefb5586cf9890f,
0x3c8eb51a92fdeffc, 0x3fefac922b7247f7,
0x3c3ebe3d702f9cd1, 0x3fefa3ec32d3d1a2,
0xbc6a033489906e0b, 0x3fef9b66affed31b,
0xbc9556522a2fbd0e, 0x3fef9301d0125b51,
0xbc5080ef8c4eea55, 0x3fef8abdc06c31cc,
0xbc91c923b9d5f416, 0x3fef829aaea92de0,
0x3c80d3e3e95c55af, 0x3fef7a98c8a58e51,
0xbc801b15eaa59348, 0x3fef72b83c7d517b,
0xbc8f1ff055de323d, 0x3fef6af9388c8dea,
0x3c8b898c3f1353bf, 0x3fef635beb6fcb75,
0xbc96d99c7611eb26, 0x3fef5be084045cd4,
0x3c9aecf73e3a2f60, 0x3fef54873168b9aa,
0xbc8fe782cb86389d, 0x3fef4d5022fcd91d,
0x3c8a6f4144a6c38d, 0x3fef463b88628cd6,
0x3c807a05b0e4047d, 0x3fef3f49917ddc96,
0x3c968efde3a8a894, 0x3fef387a6e756238,
0x3c875e18f274487d, 0x3fef31ce4fb2a63f,
0x3c80472b981fe7f2, 0x3fef2b4565e27cdd,
0xbc96b87b3f71085e, 0x3fef24dfe1f56381,
0x3c82f7e16d09ab31, 0x3fef1e9df51fdee1,
0xbc3d219b1a6fbffa, 0x3fef187fd0dad990,
0x3c8b3782720c0ab4, 0x3fef1285a6e4030b,
0x3c6e149289cecb8f, 0x3fef0cafa93e2f56,
0x3c834d754db0abb6, 0x3fef06fe0a31b715,
0x3c864201e2ac744c, 0x3fef0170fc4cd831,
0x3c8fdd395dd3f84a, 0x3feefc08b26416ff,
0xbc86a3803b8e5b04, 0x3feef6c55f929ff1,
0xbc924aedcc4b5068, 0x3feef1a7373aa9cb,
0xbc9907f81b512d8e, 0x3feeecae6d05d866,
0xbc71d1e83e9436d2, 0x3feee7db34e59ff7,
0xbc991919b3ce1b15, 0x3feee32dc313a8e5,
0x3c859f48a72a4c6d, 0x3feedea64c123422,
0xbc9312607a28698a, 0x3feeda4504ac801c,
0xbc58a78f4817895b, 0x3feed60a21f72e2a,
0xbc7c2c9b67499a1b, 0x3feed1f5d950a897,
0x3c4363ed60c2ac11, 0x3feece086061892d,
0x3c9666093b0664ef, 0x3feeca41ed1d0057,
0x3c6ecce1daa10379, 0x3feec6a2b5c13cd0,
0x3c93ff8e3f0f1230, 0x3feec32af0d7d3de,
0x3c7690cebb7aafb0, 0x3feebfdad5362a27,
0x3c931dbdeb54e077, 0x3feebcb299fddd0d,
0xbc8f94340071a38e, 0x3feeb9b2769d2ca7,
0xbc87deccdc93a349, 0x3feeb6daa2cf6642,
0xbc78dec6bd0f385f, 0x3feeb42b569d4f82,
0xbc861246ec7b5cf6, 0x3feeb1a4ca5d920f,
0x3c93350518fdd78e, 0x3feeaf4736b527da,
0x3c7b98b72f8a9b05, 0x3feead12d497c7fd,
0x3c9063e1e21c5409, 0x3feeab07dd485429,
0x3c34c7855019c6ea, 0x3feea9268a5946b7,
0x3c9432e62b64c035, 0x3feea76f15ad2148,
0xbc8ce44a6199769f, 0x3feea5e1b976dc09,
0xbc8c33c53bef4da8, 0x3feea47eb03a5585,
0xbc845378892be9ae, 0x3feea34634ccc320,
0xbc93cedd78565858, 0x3feea23882552225,
0x3c5710aa807e1964, 0x3feea155d44ca973,
0xbc93b3efbf5e2228, 0x3feea09e667f3bcd,
0xbc6a12ad8734b982, 0x3feea012750bdabf,
0xbc6367efb86da9ee, 0x3fee9fb23c651a2f,
0xbc80dc3d54e08851, 0x3fee9f7df9519484,
0xbc781f647e5a3ecf, 0x3fee9f75e8ec5f74,
0xbc86ee4ac08b7db0, 0x3fee9f9a48a58174,
0xbc8619321e55e68a, 0x3fee9feb564267c9,
0x3c909ccb5e09d4d3, 0x3feea0694fde5d3f,
0xbc7b32dcb94da51d, 0x3feea11473eb0187,
0x3c94ecfd5467c06b, 0x3feea1ed0130c132,
0x3c65ebe1abd66c55, 0x3feea2f336cf4e62,
0xbc88a1c52fb3cf42, 0x3feea427543e1a12,
0xbc9369b6f13b3734, 0x3feea589994cce13,
0xbc805e843a19ff1e, 0x3feea71a4623c7ad,
0xbc94d450d872576e, 0x3feea8d99b4492ed,
0x3c90ad675b0e8a00, 0x3feeaac7d98a6699,
0x3c8db72fc1f0eab4, 0x3feeace5422aa0db,
0xbc65b6609cc5e7ff, 0x3feeaf3216b5448c,
0x3c7bf68359f35f44, 0x3feeb1ae99157736,
0xbc93091fa71e3d83, 0x3feeb45b0b91ffc6,
0xbc5da9b88b6c1e29, 0x3feeb737b0cdc5e5,
0xbc6c23f97c90b959, 0x3feeba44cbc8520f,
0xbc92434322f4f9aa, 0x3feebd829fde4e50,
0xbc85ca6cd7668e4b, 0x3feec0f170ca07ba,
0x3c71affc2b91ce27, 0x3feec49182a3f090,
0x3c6dd235e10a73bb, 0x3feec86319e32323,
0xbc87c50422622263, 0x3feecc667b5de565,
0x3c8b1c86e3e231d5, 0x3feed09bec4a2d33,
0xbc91bbd1d3bcbb15, 0x3feed503b23e255d,
0x3c90cc319cee31d2, 0x3feed99e1330b358,
0x3c8469846e735ab3, 0x3feede6b5579fdbf,
0xbc82dfcd978e9db4, 0x3feee36bbfd3f37a,
0x3c8c1a7792cb3387, 0x3feee89f995ad3ad,
0xbc907b8f4ad1d9fa, 0x3feeee07298db666,
0xbc55c3d956dcaeba, 0x3feef3a2b84f15fb,
0xbc90a40e3da6f640, 0x3feef9728de5593a,
0xbc68d6f438ad9334, 0x3feeff76f2fb5e47,
0xbc91eee26b588a35, 0x3fef05b030a1064a,
0x3c74ffd70a5fddcd, 0x3fef0c1e904bc1d2,
0xbc91bdfbfa9298ac, 0x3fef12c25bd71e09,
0x3c736eae30af0cb3, 0x3fef199bdd85529c,
0x3c8ee3325c9ffd94, 0x3fef20ab5fffd07a,
0x3c84e08fd10959ac, 0x3fef27f12e57d14b,
0x3c63cdaf384e1a67, 0x3fef2f6d9406e7b5,
0x3c676b2c6c921968, 0x3fef3720dcef9069,
0xbc808a1883ccb5d2, 0x3fef3f0b555dc3fa,
0xbc8fad5d3ffffa6f, 0x3fef472d4a07897c,
0xbc900dae3875a949, 0x3fef4f87080d89f2,
0x3c74a385a63d07a7, 0x3fef5818dcfba487,
0xbc82919e2040220f, 0x3fef60e316c98398,
0x3c8e5a50d5c192ac, 0x3fef69e603db3285,
0x3c843a59ac016b4b, 0x3fef7321f301b460,
0xbc82d52107b43e1f, 0x3fef7c97337b9b5f,
0xbc892ab93b470dc9, 0x3fef864614f5a129,
0x3c74b604603a88d3, 0x3fef902ee78b3ff6,
0x3c83c5ec519d7271, 0x3fef9a51fbc74c83,
0xbc8ff7128fd391f0, 0x3fefa4afa2a490da,
0xbc8dae98e223747d, 0x3fefaf482d8e67f1,
0x3c8ec3bc41aa2008, 0x3fefba1bee615a27,
0x3c842b94c3a9eb32, 0x3fefc52b376bba97,
0x3c8a64a931d185ee, 0x3fefd0765b6e4540,
0xbc8e37bae43be3ed, 0x3fefdbfdad9cbe14,
0x3c77893b4d91cd9d, 0x3fefe7c1819e90d8,
0x3c5305c14160cc89, 0x3feff3c22b8f71f1,
#elif N == 256
0x0, 0x3ff0000000000000,
0xbc84e82fc61851ac, 0x3feffb1afa5abcbf,
0x3c9b3b4f1a88bf6e, 0x3feff63da9fb3335,
0xbc82985dd8521d32, 0x3feff168143b0281,
0xbc7160139cd8dc5d, 0x3fefec9a3e778061,
0x3c651e617061bfbd, 0x3fefe7d42e11bbcc,
0xbc905e7a108766d1, 0x3fefe315e86e7f85,
0x3c845fad437fa426, 0x3fefde5f72f654b1,
0x3c8cd2523567f613, 0x3fefd9b0d3158574,
0xbc954529642b232f, 0x3fefd50a0e3c1f89,
0xbc8bce8023f98efa, 0x3fefd06b29ddf6de,
0x3c8293708ef5c32e, 0x3fefcbd42b72a836,
0x3c60f74e61e6c861, 0x3fefc74518759bc8,
0xbc95b9280905b2a4, 0x3fefc2bdf66607e0,
0x3c90a3e45b33d399, 0x3fefbe3ecac6f383,
0x3c84f31f32c4b7e7, 0x3fefb9c79b1f3919,
0x3c979aa65d837b6d, 0x3fefb5586cf9890f,
0x3c9407fb30d06420, 0x3fefb0f145e46c85,
0x3c8eb51a92fdeffc, 0x3fefac922b7247f7,
0xbc9a5d04b3b9911b, 0x3fefa83b23395dec,
0x3c3ebe3d702f9cd1, 0x3fefa3ec32d3d1a2,
0xbc937a01f0739546, 0x3fef9fa55fdfa9c5,
0xbc6a033489906e0b, 0x3fef9b66affed31b,
0x3c8b8268b04ef0a5, 0x3fef973028d7233e,
0xbc9556522a2fbd0e, 0x3fef9301d0125b51,
0xbc9ac46e44a2ebcc, 0x3fef8edbab5e2ab6,
0xbc5080ef8c4eea55, 0x3fef8abdc06c31cc,
0xbc65704e90c9f860, 0x3fef86a814f204ab,
0xbc91c923b9d5f416, 0x3fef829aaea92de0,
0xbc897cea57e46280, 0x3fef7e95934f312e,
0x3c80d3e3e95c55af, 0x3fef7a98c8a58e51,
0x3c56f01429e2b9d2, 0x3fef76a45471c3c2,
0xbc801b15eaa59348, 0x3fef72b83c7d517b,
0x3c6e653b2459034b, 0x3fef6ed48695bbc0,
0xbc8f1ff055de323d, 0x3fef6af9388c8dea,
0x3c92cc7ea345b7dc, 0x3fef672658375d2f,
0x3c8b898c3f1353bf, 0x3fef635beb6fcb75,
0x3c957bfb2876ea9e, 0x3fef5f99f8138a1c,
0xbc96d99c7611eb26, 0x3fef5be084045cd4,
0x3c8cdc1873af2155, 0x3fef582f95281c6b,
0x3c9aecf73e3a2f60, 0x3fef54873168b9aa,
0xbc9493684653a131, 0x3fef50e75eb44027,
0xbc8fe782cb86389d, 0x3fef4d5022fcd91d,
0xbc98e2899077520a, 0x3fef49c18438ce4d,
0x3c8a6f4144a6c38d, 0x3fef463b88628cd6,
0x3c9120fcd4f59273, 0x3fef42be3578a819,
0x3c807a05b0e4047d, 0x3fef3f49917ddc96,
0x3c89b788c188c9b8, 0x3fef3bdda27912d1,
0x3c968efde3a8a894, 0x3fef387a6e756238,
0x3c877afbca90ef84, 0x3fef351ffb82140a,
0x3c875e18f274487d, 0x3fef31ce4fb2a63f,
0x3c91512f082876ee, 0x3fef2e85711ece75,
0x3c80472b981fe7f2, 0x3fef2b4565e27cdd,
0x3c9a02f0c7d75ec6, 0x3fef280e341ddf29,
0xbc96b87b3f71085e, 0x3fef24dfe1f56381,
0xbc803297e78260bf, 0x3fef21ba7591bb70,
0x3c82f7e16d09ab31, 0x3fef1e9df51fdee1,
0xbc95b77e5ccd9fbf, 0x3fef1b8a66d10f13,
0xbc3d219b1a6fbffa, 0x3fef187fd0dad990,
0xbc91e75c40b4251e, 0x3fef157e39771b2f,
0x3c8b3782720c0ab4, 0x3fef1285a6e4030b,
0x3c98a911f1f7785a, 0x3fef0f961f641589,
0x3c6e149289cecb8f, 0x3fef0cafa93e2f56,
0xbc61e7c998db7dbb, 0x3fef09d24abd886b,
0x3c834d754db0abb6, 0x3fef06fe0a31b715,
0x3c85425c11faadf4, 0x3fef0432edeeb2fd,
0x3c864201e2ac744c, 0x3fef0170fc4cd831,
0xbc979517a03e2847, 0x3feefeb83ba8ea32,
0x3c8fdd395dd3f84a, 0x3feefc08b26416ff,
0xbc800e2a46da4bee, 0x3feef96266e3fa2d,
0xbc86a3803b8e5b04, 0x3feef6c55f929ff1,
0xbc87430803972b34, 0x3feef431a2de883b,
0xbc924aedcc4b5068, 0x3feef1a7373aa9cb,
0xbc954de30ae02d94, 0x3feeef26231e754a,
0xbc9907f81b512d8e, 0x3feeecae6d05d866,
0xbc94f2487e1c03ec, 0x3feeea401b7140ef,
0xbc71d1e83e9436d2, 0x3feee7db34e59ff7,
0x3c914a5432fcb2f4, 0x3feee57fbfec6cf4,
0xbc991919b3ce1b15, 0x3feee32dc313a8e5,
0x3c79c3bba5562a2f, 0x3feee0e544ede173,
0x3c859f48a72a4c6d, 0x3feedea64c123422,
0xbc85a71612e21658, 0x3feedc70df1c5175,
0xbc9312607a28698a, 0x3feeda4504ac801c,
0x3c86421f6f1d24d6, 0x3feed822c367a024,
0xbc58a78f4817895b, 0x3feed60a21f72e2a,
0xbc9348a6815fce65, 0x3feed3fb2709468a,
0xbc7c2c9b67499a1b, 0x3feed1f5d950a897,
0x3c835c43984d9871, 0x3feecffa3f84b9d4,
0x3c4363ed60c2ac11, 0x3feece086061892d,
0xbc632afc8d9473a0, 0x3feecc2042a7d232,
0x3c9666093b0664ef, 0x3feeca41ed1d0057,
0xbc95fc5e44de020e, 0x3feec86d668b3237,
0x3c6ecce1daa10379, 0x3feec6a2b5c13cd0,
0xbc7ea0148327c42f, 0x3feec4e1e192aed2,
0x3c93ff8e3f0f1230, 0x3feec32af0d7d3de,
0xbc7a843ad1a88022, 0x3feec17dea6db7d7,
0x3c7690cebb7aafb0, 0x3feebfdad5362a27,
0x3c892ca3bf144e63, 0x3feebe41b817c114,
0x3c931dbdeb54e077, 0x3feebcb299fddd0d,
0xbc902c99b04aa8b0, 0x3feebb2d81d8abff,
0xbc8f94340071a38e, 0x3feeb9b2769d2ca7,
0x3c73e34f67e67118, 0x3feeb8417f4531ee,
0xbc87deccdc93a349, 0x3feeb6daa2cf6642,
0xbc75a3b1197ba0f0, 0x3feeb57de83f4eef,
0xbc78dec6bd0f385f, 0x3feeb42b569d4f82,
0x3c81bd2888075068, 0x3feeb2e2f4f6ad27,
0xbc861246ec7b5cf6, 0x3feeb1a4ca5d920f,
0xbc896be8ae89ef8f, 0x3feeb070dde910d2,
0x3c93350518fdd78e, 0x3feeaf4736b527da,
0xbc88e6ac90348602, 0x3feeae27dbe2c4cf,
0x3c7b98b72f8a9b05, 0x3feead12d497c7fd,
0xbc91af7f1365c3ac, 0x3feeac0827ff07cc,
0x3c9063e1e21c5409, 0x3feeab07dd485429,
0xbc943a3540d1898a, 0x3feeaa11fba87a03,
0x3c34c7855019c6ea, 0x3feea9268a5946b7,
0xbc951f58ddaa8090, 0x3feea84590998b93,
0x3c9432e62b64c035, 0x3feea76f15ad2148,
0xbc82e1648e50a17c, 0x3feea6a320dceb71,
0xbc8ce44a6199769f, 0x3feea5e1b976dc09,
0x3c95f30eda98a575, 0x3feea52ae6cdf6f4,
0xbc8c33c53bef4da8, 0x3feea47eb03a5585,
0x3c917ecda8a72159, 0x3feea3dd1d1929fd,
0xbc845378892be9ae, 0x3feea34634ccc320,
0xbc9345f3cee1ae6e, 0x3feea2b9febc8fb7,
0xbc93cedd78565858, 0x3feea23882552225,
0xbc85c33fdf910406, 0x3feea1c1c70833f6,
0x3c5710aa807e1964, 0x3feea155d44ca973,
0x3c81079ab5789604, 0x3feea0f4b19e9538,
0xbc93b3efbf5e2228, 0x3feea09e667f3bcd,
0x3c727df161cd7778, 0x3feea052fa75173e,
0xbc6a12ad8734b982, 0x3feea012750bdabf,
0x3c93f9924a05b767, 0x3fee9fdcddd47645,
0xbc6367efb86da9ee, 0x3fee9fb23c651a2f,
0xbc87557939a8b5ef, 0x3fee9f9298593ae5,
0xbc80dc3d54e08851, 0x3fee9f7df9519484,
0x3c51ed2f56fa9d1a, 0x3fee9f7466f42e87,
0xbc781f647e5a3ecf, 0x3fee9f75e8ec5f74,
0xbc88e67a9006c909, 0x3fee9f8286ead08a,
0xbc86ee4ac08b7db0, 0x3fee9f9a48a58174,
0x3c86597566977ac8, 0x3fee9fbd35d7cbfd,
0xbc8619321e55e68a, 0x3fee9feb564267c9,
0x3c92c0b7028a5c3a, 0x3feea024b1ab6e09,
0x3c909ccb5e09d4d3, 0x3feea0694fde5d3f,
0x3c8a30faf49cc78c, 0x3feea0b938ac1cf6,
0xbc7b32dcb94da51d, 0x3feea11473eb0187,
0xbc92dad3519d7b5b, 0x3feea17b0976cfdb,
0x3c94ecfd5467c06b, 0x3feea1ed0130c132,
0x3c87d51410fd15c2, 0x3feea26a62ff86f0,
0x3c65ebe1abd66c55, 0x3feea2f336cf4e62,
0xbc760a3629969871, 0x3feea3878491c491,
0xbc88a1c52fb3cf42, 0x3feea427543e1a12,
0x3c8b18c6e3fdef5d, 0x3feea4d2add106d9,
0xbc9369b6f13b3734, 0x3feea589994cce13,
0x3c90ec1ddcb1390a, 0x3feea64c1eb941f7,
0xbc805e843a19ff1e, 0x3feea71a4623c7ad,
0xbc522cea4f3afa1e, 0x3feea7f4179f5b21,
0xbc94d450d872576e, 0x3feea8d99b4492ed,
0x3c7c88549b958471, 0x3feea9cad931a436,
0x3c90ad675b0e8a00, 0x3feeaac7d98a6699,
0x3c931143962f7877, 0x3feeabd0a478580f,
0x3c8db72fc1f0eab4, 0x3feeace5422aa0db,
0x3c93e9e96f112479, 0x3feeae05bad61778,
0xbc65b6609cc5e7ff, 0x3feeaf3216b5448c,
0xbc8dac42a4a38df0, 0x3feeb06a5e0866d9,
0x3c7bf68359f35f44, 0x3feeb1ae99157736,
0x3c8b99dd98b1ed84, 0x3feeb2fed0282c8a,
0xbc93091fa71e3d83, 0x3feeb45b0b91ffc6,
0xbc7885ad50cbb750, 0x3feeb5c353aa2fe2,
0xbc5da9b88b6c1e29, 0x3feeb737b0cdc5e5,
0xbc82d5e85f3e0301, 0x3feeb8b82b5f98e5,
0xbc6c23f97c90b959, 0x3feeba44cbc8520f,
0xbc51669428996971, 0x3feebbdd9a7670b3,
0xbc92434322f4f9aa, 0x3feebd829fde4e50,
0x3c71f2b2c1c4c014, 0x3feebf33e47a22a2,
0xbc85ca6cd7668e4b, 0x3feec0f170ca07ba,
0xbc9294f304f166b6, 0x3feec2bb4d53fe0d,
0x3c71affc2b91ce27, 0x3feec49182a3f090,
0xbc8a1e58414c07d3, 0x3feec674194bb8d5,
0x3c6dd235e10a73bb, 0x3feec86319e32323,
0xbc79740b58a20091, 0x3feeca5e8d07f29e,
0xbc87c50422622263, 0x3feecc667b5de565,
0x3c9165830a2b96c2, 0x3feece7aed8eb8bb,
0x3c8b1c86e3e231d5, 0x3feed09bec4a2d33,
0xbc903d5cbe27874b, 0x3feed2c980460ad8,
0xbc91bbd1d3bcbb15, 0x3feed503b23e255d,
0x3c5986178980fce0, 0x3feed74a8af46052,
0x3c90cc319cee31d2, 0x3feed99e1330b358,
0xbc89472975b1f2a5, 0x3feedbfe53c12e59,
0x3c8469846e735ab3, 0x3feede6b5579fdbf,
0x3c7d8157a34b7e7f, 0x3feee0e521356eba,
0xbc82dfcd978e9db4, 0x3feee36bbfd3f37a,
0x3c8c8a4e231ebb7d, 0x3feee5ff3a3c2774,
0x3c8c1a7792cb3387, 0x3feee89f995ad3ad,
0xbc888c8d11a142e5, 0x3feeeb4ce622f2ff,
0xbc907b8f4ad1d9fa, 0x3feeee07298db666,
0x3c889c2ea41433c7, 0x3feef0ce6c9a8952,
0xbc55c3d956dcaeba, 0x3feef3a2b84f15fb,
0xbc7274aedac8ff80, 0x3feef68415b749b1,
0xbc90a40e3da6f640, 0x3feef9728de5593a,
0x3c85c620ce76df06, 0x3feefc6e29f1c52a,
0xbc68d6f438ad9334, 0x3feeff76f2fb5e47,
0xbc8fda52e1b51e41, 0x3fef028cf22749e4,
0xbc91eee26b588a35, 0x3fef05b030a1064a,
0xbc32141a7b3e2cd8, 0x3fef08e0b79a6f1f,
0x3c74ffd70a5fddcd, 0x3fef0c1e904bc1d2,
0xbc302899507554e5, 0x3fef0f69c3f3a207,
0xbc91bdfbfa9298ac, 0x3fef12c25bd71e09,
0xbc80dda2d4c0010c, 0x3fef16286141b33d,
0x3c736eae30af0cb3, 0x3fef199bdd85529c,
0xbc8a007daadf8d68, 0x3fef1d1cd9fa652c,
0x3c8ee3325c9ffd94, 0x3fef20ab5fffd07a,
0x3c836909391181d3, 0x3fef244778fafb22,
0x3c84e08fd10959ac, 0x3fef27f12e57d14b,
0xbc811cd7dbdf9547, 0x3fef2ba88988c933,
0x3c63cdaf384e1a67, 0x3fef2f6d9406e7b5,
0xbc7ac28b7bef6621, 0x3fef33405751c4db,
0x3c676b2c6c921968, 0x3fef3720dcef9069,
0xbc7030587207b9e1, 0x3fef3b0f2e6d1675,
0xbc808a1883ccb5d2, 0x3fef3f0b555dc3fa,
0xbc8cc734592af7fc, 0x3fef43155b5bab74,
0xbc8fad5d3ffffa6f, 0x3fef472d4a07897c,
0x3c87752a44f587e8, 0x3fef4b532b08c968,
0xbc900dae3875a949, 0x3fef4f87080d89f2,
0x3c85b66fefeef52e, 0x3fef53c8eacaa1d6,
0x3c74a385a63d07a7, 0x3fef5818dcfba487,
0x3c5159d9d908a96e, 0x3fef5c76e862e6d3,
0xbc82919e2040220f, 0x3fef60e316c98398,
0x3c8c254d16117a68, 0x3fef655d71ff6075,
0x3c8e5a50d5c192ac, 0x3fef69e603db3285,
0xbc8d8c329fbd0e03, 0x3fef6e7cd63a8315,
0x3c843a59ac016b4b, 0x3fef7321f301b460,
0xbc8ea6e6fbd5f2a6, 0x3fef77d5641c0658,
0xbc82d52107b43e1f, 0x3fef7c97337b9b5f,
0xbc63e8e3eab2cbb4, 0x3fef81676b197d17,
0xbc892ab93b470dc9, 0x3fef864614f5a129,
0xbc8b7966cd0d2cd9, 0x3fef8b333b16ee12,
0x3c74b604603a88d3, 0x3fef902ee78b3ff6,
0xbc776caa4c2ff1cf, 0x3fef953924676d76,
0x3c83c5ec519d7271, 0x3fef9a51fbc74c83,
0xbc81d5fc525d9940, 0x3fef9f7977cdb740,
0xbc8ff7128fd391f0, 0x3fefa4afa2a490da,
0x3c855cd8aaea3d21, 0x3fefa9f4867cca6e,
0xbc8dae98e223747d, 0x3fefaf482d8e67f1,
0x3c8269947c2bed4a, 0x3fefb4aaa2188510,
0x3c8ec3bc41aa2008, 0x3fefba1bee615a27,
0xbc83b6137e9afe9e, 0x3fefbf9c1cb6412a,
0x3c842b94c3a9eb32, 0x3fefc52b376bba97,
0xbc69fa74878ba7c7, 0x3fefcac948dd7274,
0x3c8a64a931d185ee, 0x3fefd0765b6e4540,
0x3c901f3a75ee0efe, 0x3fefd632798844f8,
0xbc8e37bae43be3ed, 0x3fefdbfdad9cbe14,
0xbc516a9ce6ed84fa, 0x3fefe1d802243c89,
0x3c77893b4d91cd9d, 0x3fefe7c1819e90d8,
0xbc699c7db2effc76, 0x3fefedba3692d514,
0x3c5305c14160cc89, 0x3feff3c22b8f71f1,
0x3c64b458677f9840, 0x3feff9d96b2a23d9,
#elif N == 512
0x0, 0x3ff0000000000000,
0xbc75d87ade1f60d5, 0x3feffd8c86da1c0a,
0xbc84e82fc61851ac, 0x3feffb1afa5abcbf,
0x3c9bffdaa7ac4bac, 0x3feff8ab5b2cbd11,
0x3c9b3b4f1a88bf6e, 0x3feff63da9fb3335,
0x3c75c18e5ae0563a, 0x3feff3d1e77170b4,
0xbc82985dd8521d32, 0x3feff168143b0281,
0xbc705b1125cf49a5, 0x3fefef003103b10e,
0xbc7160139cd8dc5d, 0x3fefec9a3e778061,
0x3c9f879abbff3f87, 0x3fefea363d42b027,
0x3c651e617061bfbd, 0x3fefe7d42e11bbcc,
0x3c9b14003824712a, 0x3fefe57411915a8a,
0xbc905e7a108766d1, 0x3fefe315e86e7f85,
0x3c61cbf0f38af658, 0x3fefe0b9b35659d8,
0x3c845fad437fa426, 0x3fefde5f72f654b1,
0xbc9a3316383dcbc5, 0x3fefdc0727fc1762,
0x3c8cd2523567f613, 0x3fefd9b0d3158574,
0x3c9901c9e0e797fd, 0x3fefd75c74f0bec2,
0xbc954529642b232f, 0x3fefd50a0e3c1f89,
0xbc89b3236d111646, 0x3fefd2b99fa6407c,
0xbc8bce8023f98efa, 0x3fefd06b29ddf6de,
0xbc8cb191be99b1b0, 0x3fefce1ead925493,
0x3c8293708ef5c32e, 0x3fefcbd42b72a836,
0xbc9acb71e83765b7, 0x3fefc98ba42e7d30,
0x3c60f74e61e6c861, 0x3fefc74518759bc8,
0x3c5cd3e58b03697e, 0x3fefc50088f8093f,
0xbc95b9280905b2a4, 0x3fefc2bdf66607e0,
0xbc8bfb07d4755452, 0x3fefc07d61701716,
0x3c90a3e45b33d399, 0x3fefbe3ecac6f383,
0x3c8aedeb3e7b14cd, 0x3fefbc02331b9715,
0x3c84f31f32c4b7e7, 0x3fefb9c79b1f3919,
0x3c9a8eb1f3d914b4, 0x3fefb78f03834e52,
0x3c979aa65d837b6d, 0x3fefb5586cf9890f,
0xbc85b9eb0402507b, 0x3fefb323d833d93f,
0x3c9407fb30d06420, 0x3fefb0f145e46c85,
0xbc93f0f225bbf3ee, 0x3fefaec0b6bdae53,
0x3c8eb51a92fdeffc, 0x3fefac922b7247f7,
0xbc9c3fe7282d1784, 0x3fefaa65a4b520ba,
0xbc9a5d04b3b9911b, 0x3fefa83b23395dec,
0x3c9c8be44bf4cde8, 0x3fefa612a7b26300,
0x3c3ebe3d702f9cd1, 0x3fefa3ec32d3d1a2,
0x3c820c5444c93c44, 0x3fefa1c7c55189c6,
0xbc937a01f0739546, 0x3fef9fa55fdfa9c5,
0xbc84c6baeb580d7a, 0x3fef9d8503328e6d,
0xbc6a033489906e0b, 0x3fef9b66affed31b,
0x3c8657aa1b0d9f83, 0x3fef994a66f951ce,
0x3c8b8268b04ef0a5, 0x3fef973028d7233e,
0x3c62f2c7fd6ee145, 0x3fef9517f64d9ef1,
0xbc9556522a2fbd0e, 0x3fef9301d0125b51,
0xbc6b0b2789925e90, 0x3fef90edb6db2dc1,
0xbc9ac46e44a2ebcc, 0x3fef8edbab5e2ab6,
0xbc93aad17d197fae, 0x3fef8ccbae51a5c8,
0xbc5080ef8c4eea55, 0x3fef8abdc06c31cc,
0xbc989c464a07ad70, 0x3fef88b1e264a0e9,
0xbc65704e90c9f860, 0x3fef86a814f204ab,
0xbc72c338fce197f4, 0x3fef84a058cbae1e,
0xbc91c923b9d5f416, 0x3fef829aaea92de0,
0xbc6dca724cea0eb6, 0x3fef809717425438,
0xbc897cea57e46280, 0x3fef7e95934f312e,
0x3c464770b955d34d, 0x3fef7c962388149e,
0x3c80d3e3e95c55af, 0x3fef7a98c8a58e51,
0xbc962811c114424f, 0x3fef789d83606e12,
0x3c56f01429e2b9d2, 0x3fef76a45471c3c2,
0x3c8ec58e74904dd4, 0x3fef74ad3c92df73,
0xbc801b15eaa59348, 0x3fef72b83c7d517b,
0x3c8d63b0ab2d5bbf, 0x3fef70c554eaea89,
0x3c6e653b2459034b, 0x3fef6ed48695bbc0,
0xbc9ca9effbeeac92, 0x3fef6ce5d23816c9,
0xbc8f1ff055de323d, 0x3fef6af9388c8dea,
0x3c8bda920de0f6e2, 0x3fef690eba4df41f,
0x3c92cc7ea345b7dc, 0x3fef672658375d2f,
0xbc9a597f9a5ff71c, 0x3fef654013041dc2,
0x3c8b898c3f1353bf, 0x3fef635beb6fcb75,
0x3c50835b125aa573, 0x3fef6179e2363cf8,
0x3c957bfb2876ea9e, 0x3fef5f99f8138a1c,
0x3c8aaa13d61aec1f, 0x3fef5dbc2dc40bf0,
0xbc96d99c7611eb26, 0x3fef5be084045cd4,
0x3c8a4f81aa7110bd, 0x3fef5a06fb91588f,
0x3c8cdc1873af2155, 0x3fef582f95281c6b,
0xbc6817fd6a313e3e, 0x3fef565a51860746,
0x3c9aecf73e3a2f60, 0x3fef54873168b9aa,
0xbc96236af85fd26a, 0x3fef52b6358e15e8,
0xbc9493684653a131, 0x3fef50e75eb44027,
0x3c7795eb4523abe7, 0x3fef4f1aad999e82,
0xbc8fe782cb86389d, 0x3fef4d5022fcd91d,
0x3c8fe58b91b40095, 0x3fef4b87bf9cda38,
0xbc98e2899077520a, 0x3fef49c18438ce4d,
0x3c91ecaa860c614a, 0x3fef47fd7190241e,
0x3c8a6f4144a6c38d, 0x3fef463b88628cd6,
0xbc3e45c83ba0bbcb, 0x3fef447bc96ffc18,
0x3c9120fcd4f59273, 0x3fef42be3578a819,
0xbc29fd3bea07b4ee, 0x3fef4102cd3d09b9,
0x3c807a05b0e4047d, 0x3fef3f49917ddc96,
0x3c87f1c7350e256d, 0x3fef3d9282fc1f27,
0x3c89b788c188c9b8, 0x3fef3bdda27912d1,
0x3c420dac6c124f4f, 0x3fef3a2af0b63bff,
0x3c968efde3a8a894, 0x3fef387a6e756238,
0xbc99501d09bc09fd, 0x3fef36cc1c78903a,
0x3c877afbca90ef84, 0x3fef351ffb82140a,
0x3c73baf864dc8675, 0x3fef33760c547f15,
0x3c875e18f274487d, 0x3fef31ce4fb2a63f,
0x3c91b0575c1eaf54, 0x3fef3028c65fa1ff,
0x3c91512f082876ee, 0x3fef2e85711ece75,
0xbc90364bc9ce33ab, 0x3fef2ce450b3cb82,
0x3c80472b981fe7f2, 0x3fef2b4565e27cdd,
0xbc7548165d85ed32, 0x3fef29a8b16f0a30,
0x3c9a02f0c7d75ec6, 0x3fef280e341ddf29,
0x3c7c3b977a68e32c, 0x3fef2675eeb3ab98,
0xbc96b87b3f71085e, 0x3fef24dfe1f56381,
0xbc93a255f697ecfe, 0x3fef234c0ea83f36,
0xbc803297e78260bf, 0x3fef21ba7591bb70,
0x3c8d2d19edc1e550, 0x3fef202b17779965,
0x3c82f7e16d09ab31, 0x3fef1e9df51fdee1,
0xbc76b2173113dd8c, 0x3fef1d130f50d65c,
0xbc95b77e5ccd9fbf, 0x3fef1b8a66d10f13,
0x3c811aa5f853590b, 0x3fef1a03fc675d1f,
0xbc3d219b1a6fbffa, 0x3fef187fd0dad990,
0x3c61d61a34c8aa02, 0x3fef16fde4f2e280,
0xbc91e75c40b4251e, 0x3fef157e39771b2f,
0xbc91f892bf6b286d, 0x3fef1400cf2f6c18,
0x3c8b3782720c0ab4, 0x3fef1285a6e4030b,
0x3c7590c65c20e680, 0x3fef110cc15d5346,
0x3c98a911f1f7785a, 0x3fef0f961f641589,
0x3c86fe320b5c1e9d, 0x3fef0e21c1c14833,
0x3c6e149289cecb8f, 0x3fef0cafa93e2f56,
0xbc903cd8b2f25790, 0x3fef0b3fd6a454d2,
0xbc61e7c998db7dbb, 0x3fef09d24abd886b,
0x3c7b3bf786a54a87, 0x3fef08670653dfe4,
0x3c834d754db0abb6, 0x3fef06fe0a31b715,
0x3c74bb6c41732885, 0x3fef05975721b004,
0x3c85425c11faadf4, 0x3fef0432edeeb2fd,
0xbc99d7399abb9a8b, 0x3fef02d0cf63eeac,
0x3c864201e2ac744c, 0x3fef0170fc4cd831,
0xbc5451d60c6ac9eb, 0x3fef001375752b40,
0xbc979517a03e2847, 0x3feefeb83ba8ea32,
0x3c8787a210ceafd9, 0x3feefd5f4fb45e20,
0x3c8fdd395dd3f84a, 0x3feefc08b26416ff,
0xbc888d1e4629943d, 0x3feefab46484ebb4,
0xbc800e2a46da4bee, 0x3feef96266e3fa2d,
0xbc93369c544088b6, 0x3feef812ba4ea77d,
0xbc86a3803b8e5b04, 0x3feef6c55f929ff1,
0x3c85373ce4eb6dfb, 0x3feef57a577dd72b,
0xbc87430803972b34, 0x3feef431a2de883b,
0x3c83adec8265a67f, 0x3feef2eb428335b4,
0xbc924aedcc4b5068, 0x3feef1a7373aa9cb,
0xbc835388bcac6bc5, 0x3feef06581d3f669,
0xbc954de30ae02d94, 0x3feeef26231e754a,
0x3c727cdb4e4b6640, 0x3feeede91be9c811,
0xbc9907f81b512d8e, 0x3feeecae6d05d866,
0x3c86c2696a26af35, 0x3feeeb761742d808,
0xbc94f2487e1c03ec, 0x3feeea401b7140ef,
0x3c888f6ff06b979a, 0x3feee90c7a61d55b,
0xbc71d1e83e9436d2, 0x3feee7db34e59ff7,
0xbc89d5efaabc2030, 0x3feee6ac4bcdf3ea,
0x3c914a5432fcb2f4, 0x3feee57fbfec6cf4,
0xbc76b8867f91c9d6, 0x3feee4559212ef89,
0xbc991919b3ce1b15, 0x3feee32dc313a8e5,
0x3c94c9c0b5157fe6, 0x3feee20853c10f28,
0x3c79c3bba5562a2f, 0x3feee0e544ede173,
0xbc62455345b51c8e, 0x3feedfc4976d27fa,
0x3c859f48a72a4c6d, 0x3feedea64c123422,
0xbc93331de45477d0, 0x3feedd8a63b0a09b,
0xbc85a71612e21658, 0x3feedc70df1c5175,
0xbc95f84d39b39b16, 0x3feedb59bf29743f,
0xbc9312607a28698a, 0x3feeda4504ac801c,
0xbc72ba4dc7c4d562, 0x3feed932b07a35df,
0x3c86421f6f1d24d6, 0x3feed822c367a024,
0xbc844f25dc02691f, 0x3feed7153e4a136a,
0xbc58a78f4817895b, 0x3feed60a21f72e2a,
0xbc888d328eb9b501, 0x3feed5016f44d8f5,
0xbc9348a6815fce65, 0x3feed3fb2709468a,
0x3c7f0bec42ddb15a, 0x3feed2f74a1af3f1,
0xbc7c2c9b67499a1b, 0x3feed1f5d950a897,
0xbc615f0a2b9cd452, 0x3feed0f6d5817663,
0x3c835c43984d9871, 0x3feecffa3f84b9d4,
0xbc8c2e465a919e1d, 0x3feecf0018321a1a,
0x3c4363ed60c2ac11, 0x3feece086061892d,
0xbc865dfd02bd08f1, 0x3feecd1318eb43ec,
0xbc632afc8d9473a0, 0x3feecc2042a7d232,
0xbc8e68cec89b1762, 0x3feecb2fde7006f4,
0x3c9666093b0664ef, 0x3feeca41ed1d0057,
0xbc48ae858eb682ca, 0x3feec9566f8827d0,
0xbc95fc5e44de020e, 0x3feec86d668b3237,
0x3c5dd71277c0915f, 0x3feec786d3001fe5,
0x3c6ecce1daa10379, 0x3feec6a2b5c13cd0,
0x3c92001325ecd7fb, 0x3feec5c10fa920a1,
0xbc7ea0148327c42f, 0x3feec4e1e192aed2,
0x3c65ace6e2870332, 0x3feec4052c5916c4,
0x3c93ff8e3f0f1230, 0x3feec32af0d7d3de,
0xbc9595c55690ffaf, 0x3feec2532feaada6,
0xbc7a843ad1a88022, 0x3feec17dea6db7d7,
0xbc8b401ba9fb5199, 0x3feec0ab213d5283,
0x3c7690cebb7aafb0, 0x3feebfdad5362a27,
0x3c6df82bf324cc57, 0x3feebf0d073537ca,
0x3c892ca3bf144e63, 0x3feebe41b817c114,
0x3c97cae38641c7bb, 0x3feebd78e8bb586b,
0x3c931dbdeb54e077, 0x3feebcb299fddd0d,
0x3c62d80c5c4a2b67, 0x3feebbeeccbd7b2a,
0xbc902c99b04aa8b0, 0x3feebb2d81d8abff,
0x3c8f39c10d12eaf0, 0x3feeba6eba2e35f0,
0xbc8f94340071a38e, 0x3feeb9b2769d2ca7,
0xbc80b582d74a55d9, 0x3feeb8f8b804f127,
0x3c73e34f67e67118, 0x3feeb8417f4531ee,
0xbc6b4e327ff434ca, 0x3feeb78ccd3deb0d,
0xbc87deccdc93a349, 0x3feeb6daa2cf6642,
0xbc592dca38593e20, 0x3feeb62b00da3b14,
0xbc75a3b1197ba0f0, 0x3feeb57de83f4eef,
0xbc85daca9994833e, 0x3feeb4d359dfd53d,
0xbc78dec6bd0f385f, 0x3feeb42b569d4f82,
0xbc980b4321bc6dae, 0x3feeb385df598d78,
0x3c81bd2888075068, 0x3feeb2e2f4f6ad27,
0xbc8390afec5241c5, 0x3feeb24298571b06,
0xbc861246ec7b5cf6, 0x3feeb1a4ca5d920f,
0x3c8f15cdafe7d586, 0x3feeb1098bed1bdf,
0xbc896be8ae89ef8f, 0x3feeb070dde910d2,
0xbc910aa91ae9b67f, 0x3feeafdac1351819,
0x3c93350518fdd78e, 0x3feeaf4736b527da,
0x3c957e1b67462375, 0x3feeaeb63f4d854c,
0xbc88e6ac90348602, 0x3feeae27dbe2c4cf,
0x3c8124d5051552a7, 0x3feead9c0d59ca07,
0x3c7b98b72f8a9b05, 0x3feead12d497c7fd,
0xbc3ca103952ecf1f, 0x3feeac8c32824135,
0xbc91af7f1365c3ac, 0x3feeac0827ff07cc,
0x3c773345c02a4fd6, 0x3feeab86b5f43d92,
0x3c9063e1e21c5409, 0x3feeab07dd485429,
0xbc909d2a0fce20f2, 0x3feeaa8b9ee20d1e,
0xbc943a3540d1898a, 0x3feeaa11fba87a03,
0xbc924f2cb4f81746, 0x3feea99af482fc8f,
0x3c34c7855019c6ea, 0x3feea9268a5946b7,
0xbc943592a0a9846b, 0x3feea8b4be135acc,
0xbc951f58ddaa8090, 0x3feea84590998b93,
0xbc956bc85d444f4f, 0x3feea7d902d47c65,
0x3c9432e62b64c035, 0x3feea76f15ad2148,
0x3c914d1e4218319f, 0x3feea707ca0cbf0f,
0xbc82e1648e50a17c, 0x3feea6a320dceb71,
0x3c971c93709313f4, 0x3feea6411b078d26,
0xbc8ce44a6199769f, 0x3feea5e1b976dc09,
0x3c7f88303b60d222, 0x3feea584fd15612a,
0x3c95f30eda98a575, 0x3feea52ae6cdf6f4,
0x3c70125ca18d4b5b, 0x3feea4d3778bc944,
0xbc8c33c53bef4da8, 0x3feea47eb03a5585,
0x3c9592ea73798b11, 0x3feea42c91c56acd,
0x3c917ecda8a72159, 0x3feea3dd1d1929fd,
0xbc9371d6d7d75739, 0x3feea390532205d8,
0xbc845378892be9ae, 0x3feea34634ccc320,
0xbc8ac05fd996f807, 0x3feea2fec30678b7,
0xbc9345f3cee1ae6e, 0x3feea2b9febc8fb7,
0xbc91f5067d03653a, 0x3feea277e8dcc390,
0xbc93cedd78565858, 0x3feea23882552225,
0x3c917339c86ce3ad, 0x3feea1fbcc140be7,
0xbc85c33fdf910406, 0x3feea1c1c70833f6,
0xbc77e66065ba2500, 0x3feea18a7420a036,
0x3c5710aa807e1964, 0x3feea155d44ca973,
0x3c964c827ee6b49a, 0x3feea123e87bfb7a,
0x3c81079ab5789604, 0x3feea0f4b19e9538,
0xbc928311a3c73480, 0x3feea0c830a4c8d4,
0xbc93b3efbf5e2228, 0x3feea09e667f3bcd,
0x3c882c79e185e981, 0x3feea077541ee718,
0x3c727df161cd7778, 0x3feea052fa75173e,
0xbc8b48cea80b043b, 0x3feea0315a736c75,
0xbc6a12ad8734b982, 0x3feea012750bdabf,
0xbc4f4863bc8e5180, 0x3fee9ff64b30aa09,
0x3c93f9924a05b767, 0x3fee9fdcddd47645,
0x3c954835dd4b7548, 0x3fee9fc62dea2f8a,
0xbc6367efb86da9ee, 0x3fee9fb23c651a2f,
0xbc8bf41f59b59f8a, 0x3fee9fa10a38cee8,
0xbc87557939a8b5ef, 0x3fee9f9298593ae5,
0xbc8f652fde52775c, 0x3fee9f86e7ba9fef,
0xbc80dc3d54e08851, 0x3fee9f7df9519484,
0xbc7b0300defbcf98, 0x3fee9f77ce1303f6,
0x3c51ed2f56fa9d1a, 0x3fee9f7466f42e87,
0xbc89dab646035dc0, 0x3fee9f73c4eaa988,
0xbc781f647e5a3ecf, 0x3fee9f75e8ec5f74,
0xbc91f0c230588dde, 0x3fee9f7ad3ef9011,
0xbc88e67a9006c909, 0x3fee9f8286ead08a,
0x3c9106450507a28c, 0x3fee9f8d02d50b8f,
0xbc86ee4ac08b7db0, 0x3fee9f9a48a58174,
0xbc9129729a10f3a0, 0x3fee9faa5953c849,
0x3c86597566977ac8, 0x3fee9fbd35d7cbfd,
0x3c781a70a5124f67, 0x3fee9fd2df29ce7c,
0xbc8619321e55e68a, 0x3fee9feb564267c9,
0x3c941626ea62646d, 0x3feea0069c1a861d,
0x3c92c0b7028a5c3a, 0x3feea024b1ab6e09,
0xbc940b9f54365b7c, 0x3feea04597eeba8f,
0x3c909ccb5e09d4d3, 0x3feea0694fde5d3f,
0x3c873455e0e826c1, 0x3feea08fda749e5d,
0x3c8a30faf49cc78c, 0x3feea0b938ac1cf6,
0x3c94f006ad874e3e, 0x3feea0e56b7fcf03,
0xbc7b32dcb94da51d, 0x3feea11473eb0187,
0xbc8f6d693d0973bb, 0x3feea14652e958aa,
0xbc92dad3519d7b5b, 0x3feea17b0976cfdb,
0x3c58c5ee2b7e7848, 0x3feea1b2988fb9ec,
0x3c94ecfd5467c06b, 0x3feea1ed0130c132,
0xbc88b25e045d207b, 0x3feea22a4456e7a3,
0x3c87d51410fd15c2, 0x3feea26a62ff86f0,
0xbc69cb3314060ca7, 0x3feea2ad5e2850ac,
0x3c65ebe1abd66c55, 0x3feea2f336cf4e62,
0x3c87a0b15d19e0bb, 0x3feea33bedf2e1b9,
0xbc760a3629969871, 0x3feea3878491c491,
0x3c94aa7212bfa73c, 0x3feea3d5fbab091f,
0xbc88a1c52fb3cf42, 0x3feea427543e1a12,
0xbc81e688272a8a12, 0x3feea47b8f4abaa9,
0x3c8b18c6e3fdef5d, 0x3feea4d2add106d9,
0x3c4ab7b7112ec9d5, 0x3feea52cb0d1736a,
0xbc9369b6f13b3734, 0x3feea589994cce13,
0x3c8a1e274eed4476, 0x3feea5e968443d9a,
0x3c90ec1ddcb1390a, 0x3feea64c1eb941f7,
0x3c94a533a59324da, 0x3feea6b1bdadb46d,
0xbc805e843a19ff1e, 0x3feea71a4623c7ad,
0x3c7a56d2760d087d, 0x3feea785b91e07f1,
0xbc522cea4f3afa1e, 0x3feea7f4179f5b21,
0x3c91682c1c6e8b05, 0x3feea86562ab00ec,
0xbc94d450d872576e, 0x3feea8d99b4492ed,
0x3c89ea99cf7a9591, 0x3feea950c27004c2,
0x3c7c88549b958471, 0x3feea9cad931a436,
0xbc59e57d8f92ff8e, 0x3feeaa47e08e1957,
0x3c90ad675b0e8a00, 0x3feeaac7d98a6699,
0x3c909b176e05a9cd, 0x3feeab4ac52be8f7,
0x3c931143962f7877, 0x3feeabd0a478580f,
0x3c711607f1952c95, 0x3feeac597875c644,
0x3c8db72fc1f0eab4, 0x3feeace5422aa0db,
0x3c869608f0f86431, 0x3feead74029db01e,
0x3c93e9e96f112479, 0x3feeae05bad61778,
0xbc7f1ced15c5c5c0, 0x3feeae9a6bdb5598,
0xbc65b6609cc5e7ff, 0x3feeaf3216b5448c,
0x3c614b97be3f7b4e, 0x3feeafccbc6c19e6,
0xbc8dac42a4a38df0, 0x3feeb06a5e0866d9,
0x3c81c1701c359530, 0x3feeb10afc931857,
0x3c7bf68359f35f44, 0x3feeb1ae99157736,
0xbc8edb1bf6809287, 0x3feeb2553499284b,
0x3c8b99dd98b1ed84, 0x3feeb2fed0282c8a,
0xbc8ba58ce7a736d3, 0x3feeb3ab6ccce12c,
0xbc93091fa71e3d83, 0x3feeb45b0b91ffc6,
0xbc93fc025e1db9ce, 0x3feeb50dad829e70,
0xbc7885ad50cbb750, 0x3feeb5c353aa2fe2,
0xbc8d737c7d71382e, 0x3feeb67bff148396,
0xbc5da9b88b6c1e29, 0x3feeb737b0cdc5e5,
0x3c6ae88c43905293, 0x3feeb7f669e2802b,
0xbc82d5e85f3e0301, 0x3feeb8b82b5f98e5,
0xbc93d1f7661fe51b, 0x3feeb97cf65253d1,
0xbc6c23f97c90b959, 0x3feeba44cbc8520f,
0x3c651b68797ffc1c, 0x3feebb0faccf9243,
0xbc51669428996971, 0x3feebbdd9a7670b3,
0x3c54579c5ceed70b, 0x3feebcae95cba768,
0xbc92434322f4f9aa, 0x3feebd829fde4e50,
0x3c87298413381667, 0x3feebe59b9bddb5b,
0x3c71f2b2c1c4c014, 0x3feebf33e47a22a2,
0xbc905000be64e965, 0x3feec01121235681,
0xbc85ca6cd7668e4b, 0x3feec0f170ca07ba,
0xbc89fb12e3454b73, 0x3feec1d4d47f2598,
0xbc9294f304f166b6, 0x3feec2bb4d53fe0d,
0x3c7be2a03697693b, 0x3feec3a4dc5a3dd3,
0x3c71affc2b91ce27, 0x3feec49182a3f090,
0x3c90622b15810eea, 0x3feec581414380f2,
0xbc8a1e58414c07d3, 0x3feec674194bb8d5,
0x3be9a5ecc875d327, 0x3feec76a0bcfc15e,
0x3c6dd235e10a73bb, 0x3feec86319e32323,
0x3c88ea486a3350ef, 0x3feec95f4499c647,
0xbc79740b58a20091, 0x3feeca5e8d07f29e,
0xbc7a2ee551d4c40f, 0x3feecb60f4424fcb,
0xbc87c50422622263, 0x3feecc667b5de565,
0x3c89c31f7e38028b, 0x3feecd6f23701b15,
0x3c9165830a2b96c2, 0x3feece7aed8eb8bb,
0xbc5fac13f4e005a3, 0x3feecf89dacfe68c,
0x3c8b1c86e3e231d5, 0x3feed09bec4a2d33,
0x3c7d8aced7162e89, 0x3feed1b1231475f7,
0xbc903d5cbe27874b, 0x3feed2c980460ad8,
0xbc848f50cea7269f, 0x3feed3e504f696b1,
0xbc91bbd1d3bcbb15, 0x3feed503b23e255d,
0x3c821eb9a08a0542, 0x3feed625893523d4,
0x3c5986178980fce0, 0x3feed74a8af46052,
0xbc6133a953131cfd, 0x3feed872b8950a73,
0x3c90cc319cee31d2, 0x3feed99e1330b358,
0x3c89e95e6f4a0ae4, 0x3feedacc9be14dca,
0xbc89472975b1f2a5, 0x3feedbfe53c12e59,
0xbc90260cf07cb311, 0x3feedd333beb0b7e,
0x3c8469846e735ab3, 0x3feede6b5579fdbf,
0x3c1bca400a7b939d, 0x3feedfa6a1897fd2,
0x3c7d8157a34b7e7f, 0x3feee0e521356eba,
0x3c9140bc34dfc19f, 0x3feee226d59a09ee,
0xbc82dfcd978e9db4, 0x3feee36bbfd3f37a,
0xbc8c9b1da461ab87, 0x3feee4b3e100301e,
0x3c8c8a4e231ebb7d, 0x3feee5ff3a3c2774,
0x3c8c115f23ebea8e, 0x3feee74dcca5a413,
0x3c8c1a7792cb3387, 0x3feee89f995ad3ad,
0xbc6dcab99f23f84e, 0x3feee9f4a17a4735,
0xbc888c8d11a142e5, 0x3feeeb4ce622f2ff,
0x3c60a43e8b7e4bfe, 0x3feeeca868742ee4,
0xbc907b8f4ad1d9fa, 0x3feeee07298db666,
0x3c915b1397075f04, 0x3feeef692a8fa8cd,
0x3c889c2ea41433c7, 0x3feef0ce6c9a8952,
0xbc839f7a1f04d2b0, 0x3feef236f0cf3f3a,
0xbc55c3d956dcaeba, 0x3feef3a2b84f15fb,
0xbc86a510f31e13e6, 0x3feef511c43bbd62,
0xbc7274aedac8ff80, 0x3feef68415b749b1,
0xbc92887ea88e7340, 0x3feef7f9ade433c6,
0xbc90a40e3da6f640, 0x3feef9728de5593a,
0xbc6e57ac604759ba, 0x3feefaeeb6ddfc87,
0x3c85c620ce76df06, 0x3feefc6e29f1c52a,
0x3c8e6c6db4f83226, 0x3feefdf0e844bfc6,
0xbc68d6f438ad9334, 0x3feeff76f2fb5e47,
0xbc8d1bf10460dba0, 0x3fef01004b3a7804,
0xbc8fda52e1b51e41, 0x3fef028cf22749e4,
0x3c8e5d80813dddfc, 0x3fef041ce8e77680,
0xbc91eee26b588a35, 0x3fef05b030a1064a,
0x3c8caff9640f2dcb, 0x3fef0746ca7a67a7,
0xbc32141a7b3e2cd8, 0x3fef08e0b79a6f1f,
0x3c7a77557fd62db3, 0x3fef0a7df9285775,
0x3c74ffd70a5fddcd, 0x3fef0c1e904bc1d2,
0xbc651ba6128db749, 0x3fef0dc27e2cb5e5,
0xbc302899507554e5, 0x3fef0f69c3f3a207,
0xbc7c0ffefdc5e251, 0x3fef111462c95b60,
0xbc91bdfbfa9298ac, 0x3fef12c25bd71e09,
0xbc8b6cd058bfd6fa, 0x3fef1473b0468d30,
0xbc80dda2d4c0010c, 0x3fef16286141b33d,
0x3c923759b8aca76d, 0x3fef17e06ff301f4,
0x3c736eae30af0cb3, 0x3fef199bdd85529c,
0xbc895498a73dac7d, 0x3fef1b5aab23e61e,
0xbc8a007daadf8d68, 0x3fef1d1cd9fa652c,
0x3c851de924583108, 0x3fef1ee26b34e065,
0x3c8ee3325c9ffd94, 0x3fef20ab5fffd07a,
0xbc8c5fe4051ba06c, 0x3fef2277b9881650,
0x3c836909391181d3, 0x3fef244778fafb22,
0xbc6d1816c0a9ac07, 0x3fef261a9f8630ad,
0x3c84e08fd10959ac, 0x3fef27f12e57d14b,
0xbc7af5c67c4e8235, 0x3fef29cb269e601f,
0xbc811cd7dbdf9547, 0x3fef2ba88988c933,
0xbc8304ef0045d575, 0x3fef2d89584661a1,
0x3c63cdaf384e1a67, 0x3fef2f6d9406e7b5,
0x3c8725f94f910375, 0x3fef31553dfa8313,
0xbc7ac28b7bef6621, 0x3fef33405751c4db,
0x3c7b53e99f9191e8, 0x3fef352ee13da7cb,
0x3c676b2c6c921968, 0x3fef3720dcef9069,
0xbc810a79e6d7e2b8, 0x3fef39164b994d23,
0xbc7030587207b9e1, 0x3fef3b0f2e6d1675,
0x3c840635f6d2a9c0, 0x3fef3d0b869d8f0f,
0xbc808a1883ccb5d2, 0x3fef3f0b555dc3fa,
0x3c549eeef9ec910c, 0x3fef410e9be12cb9,
0xbc8cc734592af7fc, 0x3fef43155b5bab74,
0xbc8335827ffb9dce, 0x3fef451f95018d17,
0xbc8fad5d3ffffa6f, 0x3fef472d4a07897c,
0x3c645563980ef762, 0x3fef493e7ba2c38c,
0x3c87752a44f587e8, 0x3fef4b532b08c968,
0xbc8cd0205eb2aab2, 0x3fef4d6b596f948c,
0xbc900dae3875a949, 0x3fef4f87080d89f2,
0xbc8aab80ceab2b4a, 0x3fef51a638197a3c,
0x3c85b66fefeef52e, 0x3fef53c8eacaa1d6,
0xbc8f870f40a8ba1b, 0x3fef55ef2158a91f,
0x3c74a385a63d07a7, 0x3fef5818dcfba487,
0x3c83c119f18464c5, 0x3fef5a461eec14be,
0x3c5159d9d908a96e, 0x3fef5c76e862e6d3,
0xbc5a628c2be4e7c7, 0x3fef5eab3a99745b,
0xbc82919e2040220f, 0x3fef60e316c98398,
0xbc72550d76be719a, 0x3fef631e7e2d479d,
0x3c8c254d16117a68, 0x3fef655d71ff6075,
0xbc82090274667d12, 0x3fef679ff37adb4a,
0x3c8e5a50d5c192ac, 0x3fef69e603db3285,
0x3c75f7d28150cac4, 0x3fef6c2fa45c4dfd,
0xbc8d8c329fbd0e03, 0x3fef6e7cd63a8315,
0x3c890de9296f4cd1, 0x3fef70cd9ab294e4,
0x3c843a59ac016b4b, 0x3fef7321f301b460,
0x3c832ff9978b34bc, 0x3fef7579e065807d,
0xbc8ea6e6fbd5f2a6, 0x3fef77d5641c0658,
0xbc7303b63dda1980, 0x3fef7a347f63c159,
0xbc82d52107b43e1f, 0x3fef7c97337b9b5f,
0xbc81f2ba385f2f95, 0x3fef7efd81a2ece1,
0xbc63e8e3eab2cbb4, 0x3fef81676b197d17,
0x3c768d9144ae12fc, 0x3fef83d4f11f8220,
0xbc892ab93b470dc9, 0x3fef864614f5a129,
0x3c853687f542403b, 0x3fef88bad7dcee90,
0xbc8b7966cd0d2cd9, 0x3fef8b333b16ee12,
0xbc736ed2de40b407, 0x3fef8daf3fe592e8,
0x3c74b604603a88d3, 0x3fef902ee78b3ff6,
0xbc614ef56c770f3b, 0x3fef92b2334ac7ee,
0xbc776caa4c2ff1cf, 0x3fef953924676d76,
0x3c8df7d1353d8e88, 0x3fef97c3bc24e350,
0x3c83c5ec519d7271, 0x3fef9a51fbc74c83,
0xbc850bed64091b8a, 0x3fef9ce3e4933c7e,
0xbc81d5fc525d9940, 0x3fef9f7977cdb740,
0x3c89d852381c317f, 0x3fefa212b6bc3181,
0xbc8ff7128fd391f0, 0x3fefa4afa2a490da,
0x3c68a00e3cca04c4, 0x3fefa7503ccd2be5,
0x3c855cd8aaea3d21, 0x3fefa9f4867cca6e,
0xbc5a1f25ce94cae7, 0x3fefac9c80faa594,
0xbc8dae98e223747d, 0x3fefaf482d8e67f1,
0xbc6fb5f3ee307976, 0x3fefb1f78d802dc2,
0x3c8269947c2bed4a, 0x3fefb4aaa2188510,
0x3c737e8ae802b851, 0x3fefb7616ca06dd6,
0x3c8ec3bc41aa2008, 0x3fefba1bee615a27,
0x3c875119560e34af, 0x3fefbcda28a52e59,
0xbc83b6137e9afe9e, 0x3fefbf9c1cb6412a,
0xbc7431c3840929c6, 0x3fefc261cbdf5be7,
0x3c842b94c3a9eb32, 0x3fefc52b376bba97,
0xbc8cb472d2e86b99, 0x3fefc7f860a70c22,
0xbc69fa74878ba7c7, 0x3fefcac948dd7274,
0x3c83f5df2fde16a8, 0x3fefcd9df15b82ac,
0x3c8a64a931d185ee, 0x3fefd0765b6e4540,
0x3c8eef18336b62e3, 0x3fefd35288633625,
0x3c901f3a75ee0efe, 0x3fefd632798844f8,
0x3c80d23f87b50a2a, 0x3fefd916302bd526,
0xbc8e37bae43be3ed, 0x3fefdbfdad9cbe14,
0x3c8302dee657c8e6, 0x3fefdee8f32a4b45,
0xbc516a9ce6ed84fa, 0x3fefe1d802243c89,
0xbc7b0caa080df170, 0x3fefe4cadbdac61d,
0x3c77893b4d91cd9d, 0x3fefe7c1819e90d8,
0x3c7617a9f2fd24e5, 0x3fefeabbf4c0ba54,
0xbc699c7db2effc76, 0x3fefedba3692d514,
0x3c75f103b8fd5ca7, 0x3feff0bc4866e8ad,
0x3c5305c14160cc89, 0x3feff3c22b8f71f1,
0x3c8e70b094fa075a, 0x3feff6cbe15f6314,
0x3c64b458677f9840, 0x3feff9d96b2a23d9,
0xbc72ec9a3e5d680a, 0x3feffceaca4391b6,
#endif
},
};
